//this file is part of eMule
//Copyright (C)2002 Merkur ( devs@emule-project.net / http://www.emule-project.net )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include "stdafx.h"
#include "emule.h"
#include "ServerConnect.h"
#include "Opcodes.h"
#include "UDPSocket.h"
#include "Exceptions.h"
#include "OtherFunctions.h"
#include "Statistics.h"
#include "ServerSocket.h"
#include "ServerList.h"
#include "Server.h"
#include "ListenSocket.h"
#include "SafeFile.h"
#include "Packets.h"
#include "SharedFileList.h"
#include "PeerCacheFinder.h"
#include "emuleDlg.h"
#include "SearchDlg.h"
#include "ServerWnd.h"
#include "TaskbarNotifier.h"
#include "Log.h"
#include "IPFilter.h"
#ifdef NATTUNNELING // NEO: UTCP - [UserModeTCP] -- Xanatos -->
#include "Neo/AbstractSocket.h"
#endif //NATTUNNELING // NEO: UTCP END <-- Xanatos --

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


// CServerConnect
// NEO: NEW - [NewCoded] -- Xanatos -->
/* David: I had some free time so I recoded this class a litle bit, it should be now better 
*                              and the memlake when shuting down while connecting should be also fixed...
*/

void CServerConnect::TryAnotherConnectionRequest()
{
	CServer* next_server;
	while((next_server = theApp.serverlist->GetNextServer(m_bTryObfuscated)) != NULL)
	{
		// Barry - Only auto-connect to static server option
		if (thePrefs.GetAutoConnectToStaticServersOnly() && !next_server->IsStaticMember())
			continue;
		if (IsLocalServer(next_server->GetIP(),next_server->GetPort(),true))
			continue;
		break;
	}

	if (next_server == NULL)
	{
		if (connectionattemps.GetCount() == 0)
		{
			if (m_bTryObfuscated && !thePrefs.IsClientCryptLayerRequired()){
				// try all servers on the non-obfuscated port next
				m_bTryObfuscated = false;
				ConnectToAnyServer(0, true, true, true);
			}
			else if (m_uOutOfServers == 0)
			{
				// 05-Nov-2003: If we have a very short server list, we could put serious load on those few servers
				// if we start the next connection tries without waiting.
				LogWarning(LOG_STATUSBAR, GetResString(IDS_OUTOFSERVERS));
				AddLogLine(false, GetResString(IDS_RECONNECT), CS_RETRYCONNECTTIME);
				m_uOutOfServers = ::GetTickCount();
				m_uStartAutoConnectPos = 0; // default: start at 0
			}
		}
		return;
	}

	ConnectToServer(next_server, true, !m_bTryObfuscated);
}

void CServerConnect::ConnectToAnyServer(UINT startAt, bool prioSort, bool isAuto, bool bNoCrypt)
{
	//StopConnectionTry();
	//Disconnect();
	connecting = true;
	if(isAuto)
		autoretry = true;
	m_uOutOfServers = 0;
	theApp.emuledlg->ShowConnectionState();
	m_bTryObfuscated = thePrefs.IsServerCryptLayerTCPRequested() && !bNoCrypt;

	// Barry - Only auto-connect to static server option
	if (thePrefs.GetAutoConnectToStaticServersOnly() /*&& isAuto*/)
	{
		bool anystatic = false;
		CServer *next_server; 
		theApp.serverlist->SetServerPosition(startAt);
		while ((next_server = theApp.serverlist->GetNextServer(false)) != NULL)
		{
			if (next_server->IsStaticMember()){
				anystatic = true;
				break;
			}
		}
		if (!anystatic){
			connecting = false;
			autoretry = false;
			LogError(LOG_STATUSBAR, GetResString(IDS_ERR_NOVALIDSERVERSFOUND));
			return;
		}
	}

	theApp.serverlist->SetServerPosition(startAt);
	if (thePrefs.GetUseUserSortedServerList() && startAt == 0 && prioSort)
		theApp.serverlist->GetUserSortedServers();
	if (thePrefs.GetUseServerPriorities() && prioSort)
		theApp.serverlist->Sort();

	// NEO: PSA - [PreferShareAll] -- Xanatos -->
	if( thePrefs.IsPreferShareAll() && prioSort )
		theApp.serverlist->PushBackNoShare();
	// NEO: PSA END <-- Xanatos --

	if (theApp.serverlist->GetServerCount() == 0) {
		connecting = false;
		LogError(LOG_STATUSBAR, GetResString(IDS_ERR_NOVALIDSERVERSFOUND));
		return;
	}
	theApp.listensocket->Process();

	TryAnotherConnectionRequest();
}

void CServerConnect::ConnectToServer(CServer* server, bool multiconnect, bool bNoCrypt)
{
	// NEO: ASFi - [AdvancedServerFilter] -- Xanatos -->
	if(theApp.ipfilter->IsFiltered(server->GetIP()))
	{
		ModLog(true,GetResString(IDS_X_SVR_IP_BANNED),ipstr(server->GetIP()), server->GetListName() );
		theApp.emuledlg->serverwnd->serverlistctrl.RemoveServer(server);
		return;
	}
	// NEO: ASFi END <-- Xanatos --

	if(!multiconnect){
		autoretry = false;
		if (IsLocalServer(server->GetIP(),server->GetPort(),true))
			return;
		{
			StopConnectionTry();
			Disconnect();
		}
	}

	connecting = true;
	theApp.emuledlg->ShowConnectionState();

	CServerSocket* newsocket = new CServerSocket(this);
#ifdef NATTUNNELING // NEO: UTCP - [UserModeTCP] -- Xanatos -->
	if(thePrefs.ReuseTCPPort())
		((CTCPSocket*)newsocket->m_Socket)->Create(thePrefs.GetPort(),SOCK_STREAM,FD_READ|FD_WRITE|FD_CLOSE|FD_CONNECT,CT2CA(theApp.GetBindAddress()),TRUE); // NEO: MOD - [BindToAdapter] <-- Xanatos --
	else
		((CTCPSocket*)newsocket->m_Socket)->Create(0,SOCK_STREAM,FD_READ|FD_WRITE|FD_CLOSE|FD_CONNECT,CT2CA(theApp.GetBindAddress())); // NEO: MOD - [BindToAdapter] <-- Xanatos --
#else
	if(thePrefs.ReuseTCPPort())
		newsocket->Create(thePrefs.GetPort(),SOCK_STREAM,FD_READ|FD_WRITE|FD_CLOSE|FD_CONNECT,CT2CA(theApp.GetBindAddress()),TRUE); // NEO: MOD - [BindToAdapter] <-- Xanatos --
	else
		newsocket->Create(0,SOCK_STREAM,FD_READ|FD_WRITE|FD_CLOSE|FD_CONNECT,CT2CA(theApp.GetBindAddress())); // NEO: MOD - [BindToAdapter] <-- Xanatos --			
#endif //NATTUNNELING // NEO: UTCP END <-- Xanatos --
	newsocket->ConnectTo(server, bNoCrypt);
	connectionattemps.AddTail(newsocket);

	return;
}

void CServerConnect::StopConnectionTry()
{
	connecting = false;
	autoretry = false;
	theApp.emuledlg->ShowConnectionState();
	m_uOutOfServers = 0;

	// close all currenty opened sockets except the one which is connected to our current server
	for(POSITION pos = connectionattemps.GetHeadPosition();pos!=NULL;){
		CServerSocket* tmpsock = connectionattemps.GetNext(pos);

		if (tcpsocket == tmpsock)		// don't destroy socket which is connected to server
			continue;
		if (tmpsock->m_bIsDeleting == false)	// don't destroy socket if it is going to destroy itself later on
			DestroySocket(tmpsock);
	}

	m_uLowIDRetrys = 0; // NEO: RLD - [ReconnectOnLowID] <-- Xanatos --
}

void CServerConnect::ConnectionEstablished(CServerSocket* sender)
{
	if (connectionattemps.IsEmpty()){
		// we are already connected to another server
		DestroySocket(sender);
		return;
	}
	
	InitLocalIP();
	if (sender->GetConnectionState() == CS_WAITFORLOGIN)
	{
#ifdef NATTUNNELING // NEO: UTCP - [UserModeTCP] -- Xanatos -->
		AddLogLine(false, GetResString(IDS_CONNECTEDTOREQ), sender->cur_server->GetListName(), sender->cur_server->GetAddress(), ((CTCPSocket*)sender->m_Socket)->IsServerCryptEnabledConnection() ? sender->cur_server->GetObfuscationPortTCP() : sender->cur_server->GetPort());
#else
		AddLogLine(false, GetResString(IDS_CONNECTEDTOREQ), sender->cur_server->GetListName(), sender->cur_server->GetAddress(), sender->IsServerCryptEnabledConnection() ? sender->cur_server->GetObfuscationPortTCP() : sender->cur_server->GetPort());
#endif //NATTUNNELING // NEO: UTCP END <-- Xanatos --

		CServer* pServer = theApp.serverlist->GetServerByAddress(sender->cur_server->GetAddress(), sender->cur_server->GetPort());
		if (pServer) {
			pServer->ResetFailedCount();
			theApp.emuledlg->serverwnd->serverlistctrl.RefreshServer(pServer);
		}

		// send loginpacket
		CSafeMemFile data(256);
		data.WriteHash16(thePrefs.GetUserHash());
		data.WriteUInt32(GetClientID());
		data.WriteUInt16(thePrefs.GetPort());

		UINT tagcount = 4;
		data.WriteUInt32(tagcount);

		CTag tagName(CT_NAME,thePrefs.GetUserNick());
		tagName.WriteTagToFile(&data);

		CTag tagVersion(CT_VERSION,EDONKEYVERSION);
		tagVersion.WriteTagToFile(&data);

		uint32 dwCryptFlags = 0;
		if (thePrefs.IsClientCryptLayerSupported())
			dwCryptFlags |= SRVCAP_SUPPORTCRYPT;
		if (thePrefs.IsClientCryptLayerRequested())
			dwCryptFlags |= SRVCAP_REQUESTCRYPT;
		if (thePrefs.IsClientCryptLayerRequired())
			dwCryptFlags |= SRVCAP_REQUIRECRYPT;

		CTag tagFlags(CT_SERVER_FLAGS, SRVCAP_ZLIB | SRVCAP_NEWTAGS | SRVCAP_LARGEFILES | SRVCAP_UNICODE | dwCryptFlags);
		tagFlags.WriteTagToFile(&data);

		// eMule Version (14-Mar-2004: requested by lugdunummaster (need for LowID clients which have no chance 
		// to send an Hello packet to the server during the callback test))
		CTag tagMuleVersion(CT_EMULE_VERSION, 
							//(uCompatibleClientID		<< 24) |
							(CemuleApp::m_nVersionMjr	<< 17) |
							(CemuleApp::m_nVersionMin	<< 10) |
							(CemuleApp::m_nVersionUpd	<<  7) );
		tagMuleVersion.WriteTagToFile(&data);

		Packet* packet = new Packet(&data);
		packet->opcode = OP_LOGINREQUEST;
		if (thePrefs.GetDebugServerTCPLevel() > 0)
			Debug(_T(">>> Sending OP__LoginRequest\n"));
		theStats.AddUpDataOverheadServer(packet->size);
		SendPacket(packet,true,sender);
	}
	else if (sender->GetConnectionState() == CS_CONNECTED)
	{
		theStats.reconnects++;
		theStats.serverConnectTime = GetTickCount();
		CString strMsg;
#ifdef NATTUNNELING // NEO: UTCP - [UserModeTCP] -- Xanatos -->
		if (((CTCPSocket*)sender->m_Socket)->IsObfusicating())
#else
		if (sender->IsObfusicating())
#endif //NATTUNNELING // NEO: UTCP END <-- Xanatos --
			strMsg.Format(GetResString(IDS_CONNECTEDTOOBFUSCATED) + _T(" (%s:%u)"), sender->cur_server->GetListName(), sender->cur_server->GetAddress(), sender->cur_server->GetObfuscationPortTCP());
		else
			strMsg.Format(GetResString(IDS_CONNECTEDTO) + _T(" (%s:%u)"), sender->cur_server->GetListName(), sender->cur_server->GetAddress(), sender->cur_server->GetPort());

		Log(LOG_SUCCESS | LOG_STATUSBAR, strMsg);
		// NEO: KLC - [KhaosLugdunumCredits] -- Xanatos -->
		// khaos::lugblistfix+ -100 for connect, we should have ~1100 right now.
		sender->cur_server->SetServerCredits(sender->cur_server->GetServerCredits() - 100);
		if(!thePrefs.UseLugdunumCredits() || (!sender->cur_server->ServerBlacklists() && thePrefs.UseLugdunumCredits() == TRUE))
		{
			DebugLog(GetResString(IDS_X_KLF_CON_ACCEPTED), sender->cur_server->GetListName());
			DebugLog(GetResString(IDS_X_KLF_CON_ACCEPTED2), sender->cur_server->GetServerCredits());
		}
		// NEO: KLC END <-- Xanatos --
		theApp.emuledlg->ShowConnectionState();
		connecting = false;
		tcpsocket = sender;
			StopConnectionTry();
		POSITION delPos = connectionattemps.Find(sender);
		if(delPos)
			connectionattemps.RemoveAt(delPos);
		theApp.sharedfiles->ClearED2KPublishInfo();
		theApp.sharedfiles->SendListToServer();
		theApp.emuledlg->serverwnd->serverlistctrl.RemoveAllDeadServers();

		// tecxx 1609 2002 - serverlist update
		if (thePrefs.GetAddServersFromServer())
		{
			Packet* packet = new Packet(OP_GETSERVERLIST,0);
			if (thePrefs.GetDebugServerTCPLevel() > 0)
				Debug(_T(">>> Sending OP__GetServerList\n"));
			theStats.AddUpDataOverheadServer(packet->size);
			SendPacket(packet,true,sender);
		}
		CServer* pServer = theApp.serverlist->GetServerByAddress(sender->cur_server->GetAddress(), sender->cur_server->GetPort());
		if (pServer)
			theApp.emuledlg->serverwnd->serverlistctrl.RefreshServer(pServer);
	}
	theApp.emuledlg->ShowConnectionState();
}

bool CServerConnect::SendPacket(Packet* packet,bool delpacket, CServerSocket* to){
	if (!to){
		if (tcpsocket){
			tcpsocket->SendPacket(packet,delpacket,true);
		}
		else{
			if (delpacket)
				delete packet;
			return false;
		}
	}
	else{
		to->SendPacket(packet,delpacket,true);
	}
	return true;
}

bool CServerConnect::SendUDPPacket(Packet* packet, CServer* host, bool delpacket, uint16 nSpecialPort, BYTE* pRawPacket, uint32 nLen){
	if (udpsocket != NULL)
		udpsocket->SendPacket(packet, host, nSpecialPort, pRawPacket, nLen);
	if (delpacket){
		delete packet;
		delete[] pRawPacket;
	}
	return true;
}

void CServerConnect::ConnectionFailed(CServerSocket* sender)
{
	if (connectionattemps.IsEmpty() && tcpsocket != sender){
		// just return, cleanup is done by the socket itself
		return;
	}
	//messages
	CServer* pServer = theApp.serverlist->GetServerByAddress(sender->cur_server->GetAddress(), sender->cur_server->GetPort());
	switch (sender->GetConnectionState())
	{
		case CS_FATALERROR:
			LogError(LOG_STATUSBAR, GetResString(IDS_ERR_FATAL));
			break;
		case CS_DISCONNECTED:
			theApp.sharedfiles->ClearED2KPublishInfo();
			LogError(LOG_STATUSBAR, GetResString(IDS_ERR_LOSTC), sender->cur_server->GetListName(), sender->cur_server->GetAddress(), sender->cur_server->GetPort());
			break;
		case CS_SERVERDEAD:
			LogError(LOG_STATUSBAR, GetResString(IDS_ERR_DEAD), sender->cur_server->GetListName(), sender->cur_server->GetAddress(), sender->cur_server->GetPort());
			if (pServer) {
				pServer->AddFailedCount();
				theApp.emuledlg->serverwnd->serverlistctrl.RefreshServer(pServer);
			}
			break;
		case CS_ERROR:
			break;
		case CS_SERVERFULL:
			LogError(LOG_STATUSBAR, GetResString(IDS_ERR_FULL), sender->cur_server->GetListName(), sender->cur_server->GetAddress(), sender->cur_server->GetPort());
			break;
		case CS_NOTCONNECTED:
			break; 
	}

	// IMPORTANT: mark this socket not to be deleted in StopConnectionTry(), 
	// because it will delete itself after this function!
	sender->m_bIsDeleting = true;

	switch (sender->GetConnectionState())
	{
		case CS_FATALERROR:{
			bool bSetRetry = thePrefs.Reconnect() && autoretry && !m_uOutOfServers;
			StopConnectionTry();
			if (bSetRetry){ 
				// There are situations where we may get Winsock error codes which indicate
				// that the network is down, although it is not. Those error codes may get
				// thrown only for particular IPs. If the first server in our list has such
				// an IP and will therefor throw such an error we would never connect to
				// any server at all. To circumvent that, start the next auto-connection
				// attempt with a different server (use the next server in the list).
				m_uStartAutoConnectPos = 0; // default: start at 0
				if (pServer) {
					// If possible, use the "next" server.
					int iPosInList = theApp.serverlist->GetPositionOfServer(pServer);
					if (iPosInList >= 0)
						m_uStartAutoConnectPos = (iPosInList + 1) % theApp.serverlist->GetServerCount();
				}

				m_uOutOfServers = ::GetTickCount();
				m_uOutOfServers += SEC2MS(CS_RETRYCONNECTTIME); // use longer interval
				LogWarning(GetResString(IDS_RECONNECT), CS_RETRYCONNECTTIME * 2);
			}
			break;
		}
		case CS_DISCONNECTED:{
			theApp.sharedfiles->ClearED2KPublishInfo();
			if (tcpsocket)
				tcpsocket->Close();
			tcpsocket = NULL;
			theApp.emuledlg->searchwnd->CancelEd2kSearch();
			// -khaos--+++> Tell our total server duration thinkymajig to update...
			theStats.serverConnectTime = 0;
			theStats.Add2TotalServerDuration();
			// <-----khaos-
			if (thePrefs.Reconnect() && !connecting)
				ConnectToAnyServer(0);		
			if (thePrefs.GetNotifierOnImportantError()) 
				theApp.emuledlg->ShowNotifier(GetResString(IDS_CONNECTIONLOST), TBN_IMPORTANTEVENT);
			break;
		}
		case CS_ERROR:
		case CS_NOTCONNECTED:{
			if (!connecting)
				break;
		}
		case CS_SERVERDEAD:
		case CS_SERVERFULL:{
			if (!connecting)
				break;
			if (!autoretry){
#ifdef NATTUNNELING // NEO: UTCP - [UserModeTCP] -- Xanatos -->
				if (pServer != NULL && ((CTCPSocket*)sender->m_Socket)->IsServerCryptEnabledConnection() && !thePrefs.IsClientCryptLayerRequired()){
#else
				if (pServer != NULL && sender->IsServerCryptEnabledConnection() && !thePrefs.IsClientCryptLayerRequired()){
#endif //NATTUNNELING // NEO: UTCP END <-- Xanatos --
					// try reconnecting without obfuscation
					ConnectToServer(pServer, false, true);
					break;
				}
				connecting = false; //StopConnectionTry();
				break;
			}

			POSITION delPos = connectionattemps.Find(sender);
			if(delPos)
				connectionattemps.RemoveAt(delPos);
		
			TryAnotherConnectionRequest();
		}
	}
	theApp.emuledlg->ShowConnectionState();
}

void CServerConnect::Process()
{
	if(IsConnecting())
		CheckForTimeout();

	if(m_uOutOfServers){
		if(GetTickCount() > m_uOutOfServers + SEC2MS(CS_RETRYCONNECTTIME)){
			StopConnectionTry();
			if (!IsConnected())
				ConnectToAnyServer();
		}
	}
	else if( autoretry && tcpsocket == NULL && (connectionattemps.GetCount() < (thePrefs.IsSafeServerConnectEnabled() ? 1 : 2)) )
		TryAnotherConnectionRequest();

	// NEO: KLC - [KhaosLugdunumCredits] -- Xanatos -->
	if (tcpsocket)
		tcpsocket->cur_server->IncServerCredits();
	// NEO: KLC END <-- Xanatos --
}

void CServerConnect::CheckForTimeout()
{ 
	DWORD dwCurTick = ::GetTickCount();
	// If we are using a proxy, increase server connection timeout to default connection timeout
	DWORD dwServerConnectTimeout = thePrefs.GetProxySettings().UseProxy ? CONNECTION_TIMEOUT : CONSERVTIMEOUT;

	for(POSITION pos = connectionattemps.GetHeadPosition();pos!=NULL;){
		CServerSocket* tmpsock = connectionattemps.GetNext(pos);
		if (dwCurTick - tmpsock->m_dwStartTime > dwServerConnectTimeout){
			LogWarning(GetResString(IDS_ERR_CONTIMEOUT), tmpsock->cur_server->GetListName(), tmpsock->cur_server->GetAddress(), tmpsock->cur_server->GetPort());
			TryAnotherConnectionRequest();
			DestroySocket(tmpsock);
			if (!autoretry)
				StopConnectionTry();
			else
				TryAnotherConnectionRequest();
		}
	}
}

bool CServerConnect::Disconnect()
{
	if(tcpsocket == NULL)
		return false;
		StopConnectionTry();
	autoretry = false;

	CServer* pServer = theApp.serverlist->GetServerByAddress(tcpsocket->cur_server->GetAddress(), tcpsocket->cur_server->GetPort());
	if (pServer)
		theApp.emuledlg->serverwnd->serverlistctrl.RefreshServer(pServer);
	DestroySocket(tcpsocket);
	tcpsocket = NULL;

	theApp.sharedfiles->ClearED2KPublishInfo();
	theApp.SetPublicIP(0);

	theApp.emuledlg->ShowConnectionState();
	// -khaos--+++> Tell our total server duration thinkymajig to update...
	theStats.serverConnectTime = 0;
	theStats.Add2TotalServerDuration();
	// <-----khaos-	

	return true;
}

CServerConnect::CServerConnect()
{
	connecting = false;
	clientid = 0;
	autoretry = false;
	udpsocket = NULL; // NEO: MOD - [BindToAdapter] <-- Xanatos --
	tcpsocket = NULL;
	m_uOutOfServers = 0;
	m_uStartAutoConnectPos = 0;
	InitLocalIP();

	m_uLowIDRetrys = 0; // NEO: RLD - [ReconnectOnLowID] <-- Xanatos --
}
// NEO: MOD - [BindToAdapter] -- Xanatos -->
bool CServerConnect::CreateUDP(){
	if(udpsocket)
		return true;

	if (thePrefs.GetServerUDPPort() != 0){
	    udpsocket = new CUDPSocket(); // initalize socket for udp packets
		if (!udpsocket->Create()){
			delete udpsocket;
			udpsocket = NULL;
			return false;
		}
	}
	return true;
}

bool CServerConnect::Rebind(){ 
	if(udpsocket) 
		return udpsocket->Rebind(); 
	return false; 
}
// NEO: MOD END <-- Xanatos --

CServerConnect::~CServerConnect(){
	// stop all connections
	StopConnectionTry();
	// close connected socket, if any
	Disconnect();
	// close udp socket
	if (udpsocket){
	    udpsocket->Close();
	    delete udpsocket;
    }
}

CServer* CServerConnect::GetCurrentServer(){
	if (IsConnected() && tcpsocket->connectionstate == CS_CONNECTED)
		return tcpsocket->cur_server;
	return NULL;
}

void CServerConnect::SetClientID(uint32 newid){
	clientid = newid;

	if (!::IsLowID(newid)){
		theApp.SetPublicIP(newid);
		// NEO: RIC - [ReaskOnIDChange] -- Xanatos -->
		if(thePrefs.IsCheckIPChange())
			theApp.CheckIDChange(newid);
		// NEO: RIC END <-- Xanatos --
	}
	
	theApp.emuledlg->ShowConnectionState();
}

void CServerConnect::DestroySocket(CServerSocket* pSck){
	if (pSck == NULL)
		return;
	// remove socket from list of opened sockets

	POSITION delPos = connectionattemps.Find(pSck);
	if(delPos)
		connectionattemps.RemoveAt(delPos);
#ifdef NATTUNNELING // NEO: UTCP - [UserModeTCP] -- Xanatos -->
	if (((CTCPSocket*)pSck->m_Socket)->m_SocketData.hSocket != INVALID_SOCKET){
#else
	if (pSck->m_SocketData.hSocket != INVALID_SOCKET){ // deadlake PROXYSUPPORT - changed to AsyncSocketEx
#endif //NATTUNNELING // NEO: UTCP END <-- Xanatos --
		pSck->AsyncSelect(0);
		pSck->Close();
	}

	delete pSck;
}

CServerSocket* CServerConnect::IsLocalServer(uint32 dwIP, uint16 nPort, bool bConnecting){
	if (IsConnected()){
		if(tcpsocket->cur_server->GetIP() == dwIP && tcpsocket->cur_server->GetPort() == nPort)
			return tcpsocket;
	}
	if(bConnecting){
		for(POSITION pos = connectionattemps.GetHeadPosition();pos!=NULL;){
			CServerSocket* cur_server = connectionattemps.GetNext(pos);
			if(cur_server->cur_server->GetIP() == dwIP && cur_server->cur_server->GetPort() == nPort)
				return cur_server;
		}
	}
	return NULL;
}

bool CServerConnect::IsConnectedServer(const CServer* server){
	if (IsConnected()){
		const CServer* cur_srv = tcpsocket->cur_server;
		if( tcpsocket->connectionstate == CS_CONNECTED
			&& cur_srv->GetPort() == server->GetPort()
			&& cur_srv->GetConnPort() == server->GetConnPort() // NEO: SAX - [ServerAUX] <-- Xanatos --
			&& _tcsicmp(cur_srv->GetAddress(), server->GetAddress()) == 0)
			return true;
	}
	return false;
}

void CServerConnect::InitLocalIP()
{
	m_nLocalIP = 0;
	// Don't use 'gethostbyname(NULL)'. The winsock DLL may be replaced by a DLL from a third party
	// which is not fully compatible to the original winsock DLL. ppl reported crash with SCORSOCK.DLL
	// when using 'gethostbyname(NULL)'.
	__try{
		char szHost[256];
		if (gethostname(szHost, sizeof szHost) == 0){
			hostent* pHostEnt = gethostbyname(szHost);
			if (pHostEnt != NULL && pHostEnt->h_length == 4 && pHostEnt->h_addr_list[0] != NULL)
				m_nLocalIP = *((uint32*)pHostEnt->h_addr_list[0]);
		}
	}
	__except(EXCEPTION_EXECUTE_HANDLER){
		// at least two ppl reported crashs when using 'gethostbyname' with third party winsock DLLs
		if (thePrefs.GetVerbose())
			DebugLogError(_T("Unknown exception in CServerConnect::InitLocalIP"));
		ASSERT(0);
	}
}

void CServerConnect::KeepConnectionAlive()
{
	DWORD dwServerKeepAliveTimeout = thePrefs.GetServerKeepAliveTimeout();

	if(tcpsocket != NULL){
		CServerSocket* cur_server = tcpsocket;
		if (dwServerKeepAliveTimeout && cur_server->connectionstate == CS_CONNECTED &&
			GetTickCount() - cur_server->GetLastTransmission() >= dwServerKeepAliveTimeout)
		{
			// "Ping" the server if the TCP connection was not used for the specified interval with 
			// an empty publish files packet -> recommended by lugdunummaster himself!
			CSafeMemFile files(4);
			files.WriteUInt32(0); // nr. of files
			Packet* packet = new Packet(&files);
			packet->opcode = OP_OFFERFILES;
			if (thePrefs.GetVerbose())
				AddDebugLogLine(false, _T("Refreshing server connection"));
			if (thePrefs.GetDebugServerTCPLevel() > 0)
				Debug(_T(">>> Sending OP__OfferFiles(KeepAlive) to server\n"));
			theStats.AddUpDataOverheadServer(packet->size);
			cur_server->SendPacket(packet,true);
		}
	}
}

bool CServerConnect::IsLowID()
{
	return ::IsLowID(clientid);
}
// NEO: NEW END <-- Xanatos --

// true if the IP is one of a server which we currently try to connect to
bool CServerConnect::AwaitingTestFromIP(uint32 dwIP) const{
	if (connectionattemps.IsEmpty())
		return false;
	CServerSocket* tmpsock;
	POSITION pos = connectionattemps.GetHeadPosition();
	while (pos) {
		tmpsock = connectionattemps.GetNext(pos);
		if (tmpsock != NULL && tmpsock->cur_server != NULL && tmpsock->cur_server->GetIP() == dwIP && tmpsock->GetConnectionState() == CS_WAITFORLOGIN)
			return true;
	}
	return false;
}

bool CServerConnect::IsConnectedObfuscated() const {
 #ifdef NATTUNNELING // NEO: UTCP - [UserModeTCP] -- Xanatos -->
	return tcpsocket != NULL && ((CTCPSocket*)tcpsocket->m_Socket)->IsObfusicating();
 #else
	return tcpsocket != NULL && tcpsocket->IsObfusicating();
 #endif //NATTUNNELING // NEO: UTCP END <-- Xanatos --
}