//this file is part of eMule
//Copyright (C)2002 Merkur ( devs@emule-project.net / http://www.emule-project.net )
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either
//version 2 of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include "stdafx.h"
#include "emule.h"
#include "ClientList.h"
#include "otherfunctions.h"
#include "Kademlia/Kademlia/kademlia.h"
#include "Kademlia/Kademlia/prefs.h"
#include "Kademlia/Kademlia/search.h"
#include "Kademlia/Kademlia/searchmanager.h"
#include "Kademlia/routing/contact.h"
#include "Kademlia/net/kademliaudplistener.h"
#include "kademlia/utils/UInt128.h"
#ifndef NEO_BC // NEO: NBC -- Xanatos -->
#include "LastCommonRouteFinder.h"
#endif // NEO_BC // NEO: NBC END <-- Xanatos --
#include "UploadQueue.h"
#include "DownloadQueue.h"
#include "UpDownClient.h"
#include "ClientCredits.h"
#include "ListenSocket.h"
#include "Opcodes.h"
#include "ServerConnect.h"
#include "emuledlg.h"
#include "TransferWnd.h"
#include "serverwnd.h"
#include "Log.h"
#include "packets.h"
#include "Statistics.h"
#include "ClientDetailDialog.h" // NEO: MLD - [ModelesDialogs] <-- Xanatos --
#ifdef ARGOS // NEO: NA - [NeoArgos] -- Xanatos -->
#include "Neo/Argos.h"
#endif // ARGOS // NEO: NA END <-- Xanatos --
#ifdef NEO_BC // NEO: NBC - [NeoBandwidthControl] -- Xanatos -->
#include "Neo/BandwidthControl/BandwidthControl.h"
#endif // NEO_BC // NEO: NBC END <-- Xanatos --
#ifdef WEBCACHE // NEO: WC - [WebCache] -- Xanatos -->
#include "PartFile.h"
#include "SharedFileList.h"
#include "Neo/WebCache/WebCache.h"
#endif // NEO: WC END <-- Xanatos --

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


CClientList::CClientList(){
	m_dwLastBannCleanUp = 0;
	m_dwLastTrackedCleanUp = 0;
	m_dwLastClientCleanUp = 0;
	m_nBuddyStatus = Disconnected;
	m_bannedList.InitHashTable(331);
	m_trackedClientsList.InitHashTable(2011);
	m_globDeadSourceList.Init(true);
	m_pBuddy = NULL;
	m_uBuddyAge = NULL; // NEO: NATS - [NatSupport] <-- Xanatos --
	m_bNotifyIdChage = false; // NEO: RIC - [ReaskOnIDChange] <-- Xanatos --
#ifdef WEBCACHE // NEO: WC - [WebCache] -- Xanatos -->
	m_dwLastSendOHCBs = 0;
#endif // NEO: WC END <-- Xanatos --
}

CClientList::~CClientList(){
	RemoveAllTrackedClients();
}

void CClientList::GetStatistics(uint32 &ruTotalClients, int stats[NUM_CLIENTLIST_STATS], 
								CMap<uint32, uint32, uint32, uint32>& clientVersionEDonkey, 
								CMap<uint32, uint32, uint32, uint32>& clientVersionEDonkeyHybrid, 
								CMap<uint32, uint32, uint32, uint32>& clientVersionEMule, 
								CMap<uint32, uint32, uint32, uint32>& clientVersionAMule)
{
	ruTotalClients = list.GetCount();
	memset(stats, 0, sizeof(stats[0]) * NUM_CLIENTLIST_STATS);

	for (POSITION pos = list.GetHeadPosition(); pos != NULL; )
	{
		const CUpDownClient* cur_client = list.GetNext(pos);

		if (cur_client->HasLowID())
			stats[14]++;
		
		switch (cur_client->GetClientSoft())
		{
			case SO_EMULE:
			case SO_OLDEMULE:
				stats[2]++;
				clientVersionEMule[cur_client->GetVersion()]++;
				break;

			case SO_EDONKEYHYBRID : 
				stats[4]++;
				clientVersionEDonkeyHybrid[cur_client->GetVersion()]++;
				break;
			
			case SO_AMULE:
				stats[10]++;
				clientVersionAMule[cur_client->GetVersion()]++;
				break;

			case SO_EDONKEY:
				stats[1]++;
				clientVersionEDonkey[cur_client->GetVersion()]++;
				break;

			case SO_MLDONKEY:
				stats[3]++;
				break;
			
			case SO_SHAREAZA:
				stats[11]++;
				break;

			// all remaining 'eMule Compatible' clients
			// NEO: ECR - [EnhancedClientRecognization] -- Xanatos -->
			case SO_HYDRANODE:
			case SO_EMULEPLUS:
			case SO_TRUSTYFILES:
			// NEO: ECR END <-- Xanatos --
			case SO_CDONKEY:
			case SO_XMULE:
			case SO_LPHANT:
				stats[5]++;
				break;

			default:
				stats[0]++;
				break;
		}

		if (cur_client->Credits() != NULL)
		{
			switch (cur_client->Credits()->GetCurrentIdentState(cur_client->GetIP()))
			{
				case IS_IDENTIFIED:
					stats[12]++;
					break;
				case IS_IDFAILED:
				case IS_IDNEEDED:
				case IS_IDBADGUY:
					stats[13]++;
					break;
			}
		}

		if (cur_client->GetDownloadState()==DS_ERROR || cur_client->GetUploadState()==US_ERROR)
			stats[6]++; // Error

		switch (cur_client->GetUserPort())
		{
			case 4662:
				stats[8]++; // Default Port
				break;
			default:
				stats[9]++; // Other Port
		}

		// Network client stats
		if (cur_client->GetServerIP() && cur_client->GetServerPort())
		{
			stats[15]++;		// eD2K
			if(cur_client->GetKadPort())
			{
				stats[17]++;	// eD2K/Kad
				stats[16]++;	// Kad
			}
		}
		else if (cur_client->GetKadPort())
			stats[16]++;		// Kad
		else
			stats[18]++;		// Unknown
	}
}

void CClientList::AddClient(CUpDownClient* toadd, bool bSkipDupTest)
{
	// skipping the check for duplicate list entries is only to be done for optimization purposes, if the calling
	// function has ensured that this client instance is not already within the list -> there are never duplicate
	// client instances in this list.
	if (!bSkipDupTest){
		if(list.Find(toadd))
			return;
	}
	theApp.emuledlg->transferwnd->clientlistctrl.AddClient(toadd);
	list.AddTail(toadd);
}

// ZZ:UploadSpeedSense -->
bool CClientList::GiveClientsForTraceRoute() {
    // this is a host that lastCommonRouteFinder can use to traceroute
#ifdef NEO_BC // NEO: NBC - [NeoBandwidthControl] -- Xanatos -->
    return theApp.bandwidthControl->AddHostsToCheck(list);
#else
    return theApp.lastCommonRouteFinder->AddHostsToCheck(list);
#endif // NEO_BC // NEO: NBC END <-- Xanatos --
}
// ZZ:UploadSpeedSense <--

void CClientList::RemoveClient(CUpDownClient* toremove, LPCTSTR pszReason){
	POSITION pos = list.Find(toremove);
	if (pos){
        theApp.uploadqueue->RemoveFromUploadQueue(toremove, CString(_T("CClientList::RemoveClient: ")) + pszReason);
		theApp.uploadqueue->RemoveFromWaitingQueue(toremove);
		theApp.downloadqueue->RemoveSource(toremove);
		theApp.emuledlg->transferwnd->clientlistctrl.RemoveClient(toremove);
		RemoveFromKadList(toremove);
		list.RemoveAt(pos);
	}
}

void CClientList::DeleteAll(){
	theApp.uploadqueue->DeleteAll();
	theApp.downloadqueue->DeleteAll();
	// NEO: MOD - [LoopImprovement] -- Xanatos -->
	while (!list.IsEmpty())
		delete list.GetHead();
	// NEO: MOD END <-- Xanatos --
}

bool CClientList::AttachToAlreadyKnown(CUpDownClient** client, CClientReqSocket* sender, EAttachMode AttachMode){ // NEO: FIX - [BetterClientAttaching] <-- Xanatos --
	CUpDownClient* tocheck = (*client);
	CUpDownClient* found_client = NULL;
	CUpDownClient* found_client2 = NULL;

	// NEO: FIX - [BetterClientAttaching] -- Xanatos -->
	if(AttachMode == ATTACH_ANSWER){
		for (POSITION pos = list.GetHeadPosition(); pos; ){ // NEO: MOD - [LoopImprovement]
			CUpDownClient* cur_client =	list.GetNext(pos);
			if (tocheck->Compare(cur_client,false)  //matching userhash
			&& !tocheck->Compare(cur_client,true)){ //different IP
				found_client = cur_client;
				break;
			}
		}
	}else
	// NEO: FIX END <-- Xanatos --
	{
		for (POSITION pos = list.GetHeadPosition(); pos; ){ // NEO: MOD - [LoopImprovement] <-- Xanatos --
			CUpDownClient* cur_client =	list.GetNext(pos); // NEO: MOD - [LoopImprovement] <-- Xanatos --
			if (tocheck->Compare(cur_client,false)){ //matching userhash
				found_client2 = cur_client;
			}
			if (tocheck->Compare(cur_client,true)){	 //matching IP
				found_client = cur_client;
				break;
			}
		}
		if (found_client == NULL)
			found_client = found_client2;
	}

	if (found_client != NULL){
		if (tocheck == found_client){
			// we found the same client instance (client may have sent more than one OP_HELLO). do not delete that client!
			// David: here the cleint already have processed the hello no need to process it 2 times
			return AttachMode != ATTACH_HALLO; // NEO: FIX - [BetterClientAttaching] <-- Xanatos --
		}
		if (sender){
			if (found_client->socket){
				if (found_client->socket->IsConnected() 
					&& (found_client->GetIP() != tocheck->GetIP() || found_client->GetUserPort() != tocheck->GetUserPort() ) )
				{
					// if found_client is connected and has the IS_IDENTIFIED, it's safe to say that the other one is a bad guy
					if (found_client->Credits() && found_client->Credits()->GetCurrentIdentState(found_client->GetIP()) == IS_IDENTIFIED){
						if (thePrefs.GetLogBannedClients())
							AddDebugLogLine(false, _T("Clients: %s (%s), Banreason: Userhash invalid"), tocheck->GetUserName(), ipstr(tocheck->GetConnectIP()));
#ifdef ARGOS // NEO: NA - [NeoArgos] -- Xanatos -->
						if(thePrefs.UseArgosSystem())
							theApp.argos->DoArgos(tocheck->GetConnectIP(),AR_HASHTHIEF,TRUE);
						else
#endif // ARGOS // NEO: NA END <-- Xanatos --
							tocheck->Ban(_T("Userhash invalid")); //Xman
						return false;
					}
	
					//IDS_CLIENTCOL Warning: Found matching client, to a currently connected client: %s (%s) and %s (%s)
					if (thePrefs.GetLogBannedClients())
						AddDebugLogLine(true,GetResString(IDS_CLIENTCOL), tocheck->GetUserName(), ipstr(tocheck->GetConnectIP()), found_client->GetUserName(), ipstr(found_client->GetConnectIP()));
					return false;
				}
				found_client->socket->client = 0;
				found_client->socket->Safe_Delete();
			}
			found_client->socket = sender;
			tocheck->socket = 0;
		}
#if defined(NEO_SK) || defined(NEO_SS) // NEO: NSK - [NeoSourceKeeper] // NEO: NSS - [NeoSourceStorage] -- Xanatos -->
		// If we find a matching clinet, that is in suspend state
		// It meen he is avalibly again, so the proper state
		if(AttachMode != ATTACH_LOAD
		 && found_client->IsSurceSuspended(true,true) && tocheck->GetDownloadState() != DS_CACHED){
			found_client->SetDownloadState(DS_REVIVAL);
		}
 #ifdef VOODOO // NEO: VOODOOx - [VoodooSourceExchange]
		// We are in offline mode we delete the first client object we got and replace it with an new one whitch have the basic informations
		else if(theApp.IsWorkingAllowed(WRK_ASKING) == false && found_client->IsSourceConfirmed() == false && !found_client->socket && thePrefs.UseVoodooSourceExchange()){
			delete found_client;
			return false;
		}
 #endif // VOODOO // NEO: VOODOOx END
#endif // NEO_SK // NEO_SS // NEO: NSK END // NEO: NSS END <-- Xanatos --
		*client = 0;
		delete tocheck;
		*client = found_client;
		return true;
	}
	return false;
}

CUpDownClient* CClientList::FindClientByIP(uint32 clientip, UINT port) const
{
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (cur_client->GetIP() == clientip && cur_client->GetUserPort() == port)
			return cur_client;
	}
	return 0;
}

CUpDownClient* CClientList::FindClientByUserHash(const uchar* clienthash) const
{
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (!md4cmp(cur_client->GetUserHash() ,clienthash))
			return cur_client;
	}
	return 0;
}

CUpDownClient* CClientList::FindClientByIP(uint32 clientip) const
{
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (cur_client->GetIP() == clientip)
			return cur_client;
	}
	return 0;
}

CUpDownClient* CClientList::FindClientByIP_UDP(uint32 clientip, UINT nUDPport) const
{
	// NEO: NATS - [NatSupport] -- Xanatos -->
	CUpDownClient* foundByIP = NULL; 
	int count = 0;
	// NEO: NATS END <-- Xanatos --
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (cur_client->GetIP() == clientip && cur_client->GetUDPPort() == nUDPport)
			return cur_client;
			// NEO: NATS - [NatSupport] -- Xanatos -->
			if (clientip == cur_client->GetIP()){
				foundByIP = cur_client;
				count++;
			}
			// NEO: NATS END <-- Xanatos --
	}
	return count == 1 ? foundByIP : 0; // NEO: NATS - [NatSupport] <-- Xanatos --
}

CUpDownClient* CClientList::FindClientByUserID_KadPort(uint32 clientID, uint16 kadPort) const
{
	// NEO: NATS - [NatSupport] -- Xanatos -->
	CUpDownClient* foundByIP = NULL; 
	int count = 0;
	// NEO: NATS END <-- Xanatos --
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (cur_client->GetUserIDHybrid() == clientID && cur_client->GetKadPort() == kadPort)
			return cur_client;
		// NEO: NATS - [NatSupport] -- Xanatos -->
		if (clientID == cur_client->GetUserIDHybrid()){
			foundByIP = cur_client;
			count++;
		}
		// NEO: NATS END <-- Xanatos --
	}
	return count == 1 ? foundByIP : 0; // NEO: NATS - [NatSupport] <-- Xanatos --
}

CUpDownClient* CClientList::FindClientByIP_KadPort(uint32 ip, uint16 port) const
{
	// NEO: NATS - [NatSupport] -- Xanatos -->
	CUpDownClient* foundByIP = NULL; 
	int count = 0;
	// NEO: NATS END <-- Xanatos --
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (cur_client->GetIP() == ip && cur_client->GetKadPort() == port)
			return cur_client;
		// NEO: NATS - [NatSupport] -- Xanatos -->
		if (ip == cur_client->GetIP()){
			foundByIP = cur_client;
			count++;
		}
		// NEO: NATS END <-- Xanatos --
	}
	return count == 1 ? foundByIP : 0; // NEO: NATS - [NatSupport] <-- Xanatos --
}

#ifdef NATTUNNELING // NEO: NATT - [NatTraversal] -- Xanatos -->
CUpDownClient* CClientList::FindClientByBuddyID(const uchar* BuddyID) const
{
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (!md4cmp(cur_client->GetBuddyID(), BuddyID))
			return cur_client;
	}
	return 0;
}

CUpDownClient* CClientList::FindClientByNatIP_NatPort(uint32 natIP, uint16 natPort) const
{
	// NEO: NATS - [NatSupport]
	CUpDownClient* foundByIP = NULL; 
	int count = 0;
	// NEO: NATS END
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if (cur_client->GetConnectIP() == natIP && cur_client->GetNatPort() == natPort)
			return cur_client;
		// NEO: NATS - [NatSupport]
		if (natIP == cur_client->GetConnectIP()){
			foundByIP = cur_client;
			count++;
		}
		// NEO: NATS END
	}
	return count == 1 ? foundByIP : 0; // NEO: NATS - [NatSupport]
}

#endif //NATTUNNELING // NEO: NATT END <-- Xanatos --

CUpDownClient* CClientList::FindClientByServerID(uint32 uServerIP, uint32 uED2KUserID) const
{
	uint32 uHybridUserID = ntohl(uED2KUserID);
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client =	list.GetNext(pos);
		if (cur_client->GetServerIP() == uServerIP && cur_client->GetUserIDHybrid() == uHybridUserID)
			return cur_client;
	}
	return 0;
}


///////////////////////////////////////////////////////////////////////////////
// Banned clients

void CClientList::AddBannedClient(uint32 dwIP){
	m_bannedList.SetAt(dwIP, ::GetTickCount());
}

bool CClientList::IsBannedClient(uint32 dwIP) const
{
	uint32 dwBantime;
	if (m_bannedList.Lookup(dwIP, dwBantime)){
#ifdef ARGOS // NEO: NA - [NeoArgos] -- Xanatos -->
		if (dwBantime + thePrefs.GetArgosGenericBanTimeMs() > ::GetTickCount())
#else
		if (dwBantime + CLIENTBANTIME > ::GetTickCount())
#endif // ARGOS // NEO: NA END <-- Xanatos --

			return true;
	}
	return false; 
}

void CClientList::RemoveBannedClient(uint32 dwIP){
	m_bannedList.RemoveKey(dwIP);
}

void CClientList::RemoveAllBannedClients(){
	m_bannedList.RemoveAll();
}


///////////////////////////////////////////////////////////////////////////////
// Tracked clients

void CClientList::AddTrackClient(CUpDownClient* toadd){
	CDeletedClient* pResult = 0;
	if (m_trackedClientsList.Lookup(toadd->GetIP(), pResult)){
		pResult->m_dwInserted = ::GetTickCount();
		for (int i = 0; i != pResult->m_ItemsList.GetCount(); i++){
			if (pResult->m_ItemsList[i].nPort == toadd->GetUserPort()){
				// already tracked, update
				//pResult->m_ItemsList[i].pHash = toadd->Credits(); //Xman don't keep a track of the credit-pointer, but of the hash
				md4cpy(pResult->m_ItemsList[i].pHash, toadd->GetUserHash()); // NEO: XCTA - [XmanExtenedCreditTableArragement] <-- Xanatos --
				return;
			}
		}
		//PORTANDHASH porthash = { toadd->GetUserPort(), toadd->Credits()};
		// NEO: XCTA - [XmanExtenedCreditTableArragement] -- Xanatos -->
		//Xman new tracked port & hash
		PORTANDHASH porthash;
		porthash.nPort=toadd->GetUserPort();
		md4cpy(porthash.pHash,toadd->GetUserHash());
		// NEO: XCTA END <-- Xanatos --
		pResult->m_ItemsList.Add(porthash);
	}
	else{
		m_trackedClientsList.SetAt(toadd->GetIP(), new CDeletedClient(toadd));
	}
}

// true = everything ok, hash didn't changed
// false = hash changed
bool CClientList::ComparePriorUserhash(uint32 dwIP, uint16 nPort, void* pNewHash){
	CDeletedClient* pResult = 0;
	if (m_trackedClientsList.Lookup(dwIP, pResult)){
		for (int i = 0; i != pResult->m_ItemsList.GetCount(); i++){
			if (pResult->m_ItemsList[i].nPort == nPort){
				//if (pResult->m_ItemsList[i].pHash != pNewHash)
				if (md4cmp(pResult->m_ItemsList[i].pHash , pNewHash)!=0) // NEO: XCTA - [XmanExtenedCreditTableArragement] <-- Xanatos --
					return false;
				else
					break;
			}
		}
	}
	return true;
}

UINT CClientList::GetClientsFromIP(uint32 dwIP) const
{
	CDeletedClient* pResult;
	if (m_trackedClientsList.Lookup(dwIP, pResult))
		return pResult->m_ItemsList.GetCount();
	return 0;
}

void CClientList::TrackBadRequest(const CUpDownClient* upcClient, int nIncreaseCounter){
	CDeletedClient* pResult = NULL;
	if (upcClient->GetIP() == 0){
		ASSERT( false );
		return;
	}
	if (m_trackedClientsList.Lookup(upcClient->GetIP(), pResult)){
		pResult->m_dwInserted = ::GetTickCount();
		pResult->m_cBadRequest += nIncreaseCounter;
	}
	else{
		CDeletedClient* ccToAdd = new CDeletedClient(upcClient);
		ccToAdd->m_cBadRequest = nIncreaseCounter;
		m_trackedClientsList.SetAt(upcClient->GetIP(), ccToAdd);
	}
}

uint32 CClientList::GetBadRequests(const CUpDownClient* upcClient) const{
	CDeletedClient* pResult = NULL;
	if (upcClient->GetIP() == 0){
		ASSERT( false );
		return 0;
	}
	if (m_trackedClientsList.Lookup(upcClient->GetIP(), pResult)){
		return pResult->m_cBadRequest;
	}
	else
		return 0;
}

void CClientList::RemoveAllTrackedClients(){
	POSITION pos = m_trackedClientsList.GetStartPosition();
	uint32 nKey;
	CDeletedClient* pResult;
	while (pos != NULL){
		m_trackedClientsList.GetNextAssoc(pos, nKey, pResult);
		m_trackedClientsList.RemoveKey(nKey);
		delete pResult;
	}
}

void CClientList::Process()
{
	const uint32 cur_tick = ::GetTickCount();
	///////////////////////////////////////////////////////////////////////////
	// Cleanup banned client list
	//
#ifdef ARGOS // NEO: NA - [NeoArgos] -- Xanatos -->
	if(!thePrefs.UseArgosSystem()) // when the argos is enabled it cleans the list for its own
		if (m_dwLastBannCleanUp + thePrefs.GetArgosProcessTimeMs(true) < cur_tick)
#else
		if (m_dwLastBannCleanUp + BAN_CLEANUP_TIME < cur_tick)
#endif // ARGOS // NEO: NA END <-- Xanatos --
		{
			m_dwLastBannCleanUp = cur_tick;
			
			POSITION pos = m_bannedList.GetStartPosition();
			uint32 nKey;
			uint32 dwBantime;
			while (pos != NULL)
			{
				m_bannedList.GetNextAssoc( pos, nKey, dwBantime );
#ifdef ARGOS // NEO: NA - [NeoArgos] -- Xanatos -->
				if (dwBantime + thePrefs.GetArgosGenericBanTimeMs() < cur_tick )
#else
				if (dwBantime + CLIENTBANTIME < cur_tick )
#endif // ARGOS // NEO: NA END <-- Xanatos --
					RemoveBannedClient(nKey);
			}
		}


	///////////////////////////////////////////////////////////////////////////
	// Cleanup tracked client list
	//
#ifdef ARGOS // NEO: NA - [NeoArgos] -- Xanatos -->
	if (m_dwLastTrackedCleanUp + thePrefs.GetArgosProcessTimeMs() < cur_tick)
#else
	if (m_dwLastTrackedCleanUp + TRACKED_CLEANUP_TIME < cur_tick
#endif // ARGOS // NEO: NA END <-- Xanatos --
	{
		m_dwLastTrackedCleanUp = cur_tick;
		if (thePrefs.GetLogBannedClients())
			AddDebugLogLine(false, _T("Cleaning up TrackedClientList, %i clients on List..."), m_trackedClientsList.GetCount());
		POSITION pos = m_trackedClientsList.GetStartPosition();
		uint32 nKey;
		CDeletedClient* pResult;
		while (pos != NULL)
		{
			m_trackedClientsList.GetNextAssoc( pos, nKey, pResult );
#ifdef ARGOS // NEO: NA - [NeoArgos] -- Xanatos -->
			if (pResult->m_dwInserted + thePrefs.GetArgosTrackTimeMs() < cur_tick ){
#else
			if (pResult->m_dwInserted + KEEPTRACK_TIME < cur_tick ){
#endif // ARGOS // NEO: NA END <-- Xanatos --
				m_trackedClientsList.RemoveKey(nKey);
				delete pResult;
			}
		}
		if (thePrefs.GetLogBannedClients())
			AddDebugLogLine(false, _T("...done, %i clients left on list"), m_trackedClientsList.GetCount());
	}

	///////////////////////////////////////////////////////////////////////////
	// Process Kad client list
	//
	//We need to try to connect to the clients in m_KadList
	//If connected, remove them from the list and send a message back to Kad so we can send a ACK.
	//If we don't connect, we need to remove the client..
	//The sockets timeout should delete this object.


	// buddy is just a flag that is used to make sure we are still connected or connecting to a buddy.
	buddyState buddy = Disconnected;

	for (POSITION pos = m_KadList.GetHeadPosition(); pos; ) // NEO: MOD - [LoopImprovement] <-- Xanatos --
	{
		CUpDownClient* cur_client =	m_KadList.GetNext(pos); // NEO: MOD - [LoopImprovement] <-- Xanatos --
		if( !Kademlia::CKademlia::IsRunning() )
		{
			//Clear out this list if we stop running Kad.
			//Setting the Kad state to KS_NONE causes it to be removed in the switch below.
			cur_client->SetKadState(KS_NONE);
		}
		switch(cur_client->GetKadState())
		{
			case KS_QUEUED_FWCHECK:
				//Another client asked us to try to connect to them to check their firewalled status.
				cur_client->TryToConnect(true);
				break;

			case KS_CONNECTING_FWCHECK:
				//Ignore this state as we are just waiting for results.
				break;

			case KS_CONNECTED_FWCHECK:
				//We successfully connected to the client.
				//We now send a ack to let them know.
				if (thePrefs.GetDebugClientKadUDPLevel() > 0)
					DebugSend("KADEMLIA_FIREWALLED_ACK_RES", cur_client->GetIP(), cur_client->GetKadPort());
				Kademlia::CKademlia::GetUDPListener()->SendNullPacket(KADEMLIA_FIREWALLED_ACK_RES, ntohl(cur_client->GetIP()), cur_client->GetKadPort());
				//We are done with this client. Set Kad status to KS_NONE and it will be removed in the next cycle.
				cur_client->SetKadState(KS_NONE);
				break;

			case KS_INCOMING_BUDDY:
				//A firewalled client wants us to be his buddy.
				//If we already have a buddy, we set Kad state to KS_NONE and it's removed in the next cycle.
				//If not, this client will change to KS_CONNECTED_BUDDY when it connects.
				if( m_nBuddyStatus == Connected )
					cur_client->SetKadState(KS_NONE);
				break;

			case KS_QUEUED_BUDDY:
				//We are firewalled and want to request this client to be a buddy.
				//But first we check to make sure we are not already trying another client.
				//If we are not already trying. We try to connect to this client.
				//If we are already connected to a buddy, we set this client to KS_NONE and it's removed next cycle.
				//If we are trying to connect to a buddy, we just ignore as the one we are trying may fail and we can then try this one.
				if( m_nBuddyStatus == Disconnected )
				{
					buddy = Connecting;
					m_nBuddyStatus = Connecting;
					cur_client->SetKadState(KS_CONNECTING_BUDDY);
					cur_client->TryToConnect(true);
					theApp.emuledlg->serverwnd->UpdateMyInfo();
				}
				else if( m_nBuddyStatus == Connected )
					cur_client->SetKadState(KS_NONE);
				break;

			case KS_CONNECTING_BUDDY:
				//We are trying to connect to this client.
				//Although it should NOT happen, we make sure we are not already connected to a buddy.
				//If we are we set to KS_NONE and it's removed next cycle.
				//But if we are not already connected, make sure we set the flag to connecting so we know 
				//things are working correctly.
				if( m_nBuddyStatus == Connected )
					cur_client->SetKadState(KS_NONE);
				else
				{
					ASSERT( m_nBuddyStatus == Connecting );
					buddy = Connecting;
				}
				break;

			case KS_CONNECTED_BUDDY:
				//A potential connected buddy client wanting to me in the Kad network
				//We set our flag to connected to make sure things are still working correctly.
				buddy = Connected;
				
				//If m_nBuddyStatus is not connected already, we set this client as our buddy!
				if( m_nBuddyStatus != Connected )
				{
					m_pBuddy = cur_client;
					m_nBuddyStatus = Connected;
					m_uBuddyAge = ::GetTickCount(); // NEO: NATS - [NatSupport] <-- Xanatos --
					theApp.emuledlg->serverwnd->UpdateMyInfo();
				}
				if( m_pBuddy == cur_client && theApp.IsFirewalled() && cur_client->SendBuddyPingPong() )
				{
					if (thePrefs.GetDebugClientTCPLevel() > 0)
						DebugSend("OP__BuddyPing", cur_client);
					Packet* buddyPing = new Packet(OP_BUDDYPING, 0, OP_EMULEPROT);
					theStats.AddUpDataOverheadOther(buddyPing->size);
					cur_client->SafeSendPacket(buddyPing);
					cur_client->SetLastBuddyPingPongTime();
				}
				break;

			default:
				RemoveFromKadList(cur_client);
		}
	}
	
	//We either never had a buddy, or lost our buddy..
	if( buddy == Disconnected )
	{
		if( m_nBuddyStatus != Disconnected || m_pBuddy )
		{
			if( Kademlia::CKademlia::IsRunning() && theApp.IsFirewalled() )
			{
				//We are a lowID client and we just lost our buddy.
				//Go ahead and instantly try to find a new buddy.
				Kademlia::CKademlia::GetPrefs()->SetFindBuddy();
			}
			m_pBuddy = NULL;
			m_nBuddyStatus = Disconnected;
			m_uBuddyAge = 0; // NEO: NATS - [NatSupport] <-- Xanatos --
			theApp.emuledlg->serverwnd->UpdateMyInfo();
		}
	}

	if ( Kademlia::CKademlia::IsConnected() )
	{
		if( Kademlia::CKademlia::IsFirewalled() )
		{
			//TODO: Kad buddies won'T work with RequireCrypt, so it is disabled for now but should (and will)
			//be fixed in later version
			if( m_nBuddyStatus == Disconnected && Kademlia::CKademlia::GetPrefs()->GetFindBuddy() && !thePrefs.IsClientCryptLayerRequired())
			{
				//We are a firewalled client with no buddy. We have also waited a set time 
				//to try to avoid a false firewalled status.. So lets look for a buddy..
				if( !Kademlia::CSearchManager::PrepareLookup(Kademlia::CSearch::FINDBUDDY, true, Kademlia::CUInt128(true).Xor(Kademlia::CKademlia::GetPrefs()->GetKadID())) )
				{
					//This search ID was already going. Most likely reason is that
					//we found and lost our buddy very quickly and the last search hadn't
					//had time to be removed yet. Go ahead and set this to happen again
					//next time around.
					Kademlia::CKademlia::GetPrefs()->SetFindBuddy();
				}
			}
		}
		else
		{
			if( m_pBuddy )
			{
				//Lets make sure that if we have a buddy, they are firewalled!
				//If they are also not firewalled, then someone must have fixed their firewall or stopped saturating their line.. 
				//We just set the state of this buddy to KS_NONE and things will be cleared up with the next cycle.
				if( !m_pBuddy->HasLowID() )
					m_pBuddy->SetKadState(KS_NONE);
			}
		}
	}
	else
	{
		if( m_pBuddy )
		{
			//We are not connected anymore. Just set this buddy to KS_NONE and things will be cleared out on next cycle.
			m_pBuddy->SetKadState(KS_NONE);
		}
	}

	// NEO: RIC - [ReaskOnIDChange] -- Xanatos -->
	if(m_bNotifyIdChage)
	{
		bool bNotDone = false;
		for(POSITION pos = list.GetHeadPosition(); pos != NULL;){				
			CUpDownClient* cur_client =	list.GetNext(pos);
			if(cur_client->WaitingNotifyIdChage()){
				if(cur_client->NotifyIdChage())
					bNotDone = true;
			}
		}
		if(!bNotDone)
			m_bNotifyIdChage = false;
	}
	// NEO: RIC END <-- Xanatos --

	///////////////////////////////////////////////////////////////////////////
	// Cleanup client list
	//
	CleanUpClientList();
#ifdef WEBCACHE // NEO: WC - [WebCache] -- Xanatos -->
	if(thePrefs.IsWebCacheEnabled())
		SendOHCBs();
#endif // NEO: WC END <-- Xanatos --
}

#ifdef _DEBUG
void CClientList::Debug_SocketDeleted(CClientReqSocket* deleted) const
{
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;){
		CUpDownClient* cur_client =	list.GetNext(pos);
		if (!AfxIsValidAddress(cur_client, sizeof(CUpDownClient))) {
			AfxDebugBreak();
		}
		if (thePrefs.m_iDbgHeap >= 2)
			ASSERT_VALID(cur_client);
		if (cur_client->socket == deleted){
			AfxDebugBreak();
		}
	}
}
#endif

bool CClientList::IsValidClient(CUpDownClient* tocheck) const
{
	if (thePrefs.m_iDbgHeap >= 2)
		ASSERT_VALID(tocheck);
	return list.Find(tocheck)!=NULL;
}


///////////////////////////////////////////////////////////////////////////////
// Kad client list

void CClientList::RequestTCP(Kademlia::CContact* contact)
{
	uint32 nContactIP = ntohl(contact->GetIPAddress());
	// don't connect ourself
	if (theApp.serverconnect->GetLocalIP() == nContactIP && thePrefs.GetPort() == contact->GetTCPPort())
		return;

	CUpDownClient* pNewClient = FindClientByIP(nContactIP, contact->GetTCPPort());

	const bool bNewClient = pNewClient == NULL; // NEO: MOD - [CodeImprovement] <-- Xanatos -- //Xman Code Improvement don't search new generated clients in lists (seen by Wizard)

	if (!pNewClient)
		pNewClient = new CUpDownClient(0, contact->GetTCPPort(), contact->GetIPAddress(), 0, 0, false );

	//Add client to the lists to be processed.
	pNewClient->SetKadPort(contact->GetUDPPort());
	pNewClient->SetKadState(KS_QUEUED_FWCHECK);

	// NEO: MOD - [CodeImprovement] -- Xanatos -->
	//Xman Code Improvement don't search new generated clients in lists (seen by Wizard)
	//Xman no need to check for dupe in clientlist, either we found it or it's new
	//if not new do a dupe check in kad-list and don't add to clientlist
	if(bNewClient)
	{
		m_KadList.AddTail(pNewClient);
		AddClient(pNewClient, true); 
	}
	else
		AddToKadList(pNewClient); 
	// NEO: MOD END <-- Xanatos --
	//m_KadList.AddTail(pNewClient);
	//This method checks if this is a dup already.
	//AddClient(pNewClient);
}

void CClientList::RequestBuddy(Kademlia::CContact* contact)
{
	uint32 nContactIP = ntohl(contact->GetIPAddress());
	// don't connect ourself
	if (theApp.serverconnect->GetLocalIP() == nContactIP && thePrefs.GetPort() == contact->GetTCPPort())
		return;
	CUpDownClient* pNewClient = FindClientByIP(nContactIP, contact->GetTCPPort());

	const bool bNewClient = pNewClient == NULL; // NEO: MOD - [CodeImprovement] <-- Xanatos -- //Xman Code Improvement don't search new generated clients in lists (seen by Wizard)

	if (!pNewClient)
		pNewClient = new CUpDownClient(0, contact->GetTCPPort(), contact->GetIPAddress(), 0, 0, false );

	//Add client to the lists to be processed.
	pNewClient->SetKadPort(contact->GetUDPPort());
	pNewClient->SetKadState(KS_QUEUED_BUDDY);
	byte ID[16];
	contact->GetClientID().ToByteArray(ID);
	pNewClient->SetUserHash(ID);

	// NEO: MOD - [CodeImprovement] -- Xanatos -->
	//Xman no need to check for dupe in clientlist, either we found it or it's new
	//if not new do a dupe check in kad-list and don't add to clientlist
	if(bNewClient)
	{
		m_KadList.AddTail(pNewClient);
		AddClient(pNewClient, true); 
	}
	else
		AddToKadList(pNewClient); 
	// NEO: MOD END <-- Xanatos --

	//AddToKadList(pNewClient);
	//This method checks if this is a dup already.
	//AddClient(pNewClient);
}

void CClientList::IncomingBuddy(Kademlia::CContact* contact, Kademlia::CUInt128* buddyID )
{
	uint32 nContactIP = ntohl(contact->GetIPAddress());
	//If eMule already knows this client, abort this.. It could cause conflicts.
	//Although the odds of this happening is very small, it could still happen.
	if (FindClientByIP(nContactIP, contact->GetTCPPort()))
	{
		return;
	}

	// don't connect ourself
	if (theApp.serverconnect->GetLocalIP() == nContactIP && thePrefs.GetPort() == contact->GetTCPPort())
		return;

	//Add client to the lists to be processed.
	CUpDownClient* pNewClient = new CUpDownClient(0, contact->GetTCPPort(), contact->GetIPAddress(), 0, 0, false );
	pNewClient->SetKadPort(contact->GetUDPPort());
	pNewClient->SetKadState(KS_INCOMING_BUDDY);
	byte ID[16];
	contact->GetClientID().ToByteArray(ID);
	pNewClient->SetUserHash(ID);
	buddyID->ToByteArray(ID);
	pNewClient->SetBuddyID(ID);

	// NEO: MOD - [CodeImprovement] -- Xanatos -->
	//Xman Code Improvement don't search new generated clients in lists (seen by Wizard)
	//Xman it's a new client -> no dupe check
	m_KadList.AddTail(pNewClient);
	AddClient(pNewClient, true); 
	// NEO: MOD END <-- Xanatos --

	//AddToKadList(pNewClient);
	//AddClient(pNewClient);
}

void CClientList::RemoveFromKadList(CUpDownClient* torem){
	POSITION pos = m_KadList.Find(torem);
	if(pos)
	{
		if(torem == m_pBuddy)
		{
			m_pBuddy = NULL;
			m_uBuddyAge = 0; // NEO: NATS - [NatSupport] <-- Xanatos --
			theApp.emuledlg->serverwnd->UpdateMyInfo();
		}
		m_KadList.RemoveAt(pos);
	}
}

void CClientList::AddToKadList(CUpDownClient* toadd){
	if(!toadd)
		return;
	POSITION pos = m_KadList.Find(toadd);
	if(pos)
	{
		return;
	}
	m_KadList.AddTail(toadd);
}

void CClientList::CleanUpClientList(){
	// we remove clients which are not needed any more by time
	// this check is also done on CUpDownClient::Disconnected, however it will not catch all
	// cases (if a client changes the state without beeing connected
	//
	// Adding this check directly to every point where any state changes would be more effective,
	// is however not compatible with the current code, because there are points where a client has
	// no state for some code lines and the code is also not prepared that a client object gets
	// invalid while working with it (aka setting a new state)
	// so this way is just the easy and safe one to go (as long as emule is basically single threaded)
	const uint32 cur_tick = ::GetTickCount();
	if (m_dwLastClientCleanUp + CLIENTLIST_CLEANUP_TIME < cur_tick ){
		m_dwLastClientCleanUp = cur_tick;
		uint32 cDeleted = 0;
		for (POSITION pos = list.GetHeadPosition(); pos;){ // NEO: MOD - [LoopImprovement] <-- Xanatos --
			CUpDownClient* pCurClient =	list.GetNext(pos); // NEO: MOD - [LoopImprovement] <-- Xanatos --
			if ((pCurClient->GetUploadState() == US_NONE || pCurClient->GetUploadState() == US_BANNED && !pCurClient->IsBanned())
				&& pCurClient->GetDownloadState() == DS_NONE
				&& pCurClient->GetChatState() == MS_NONE
				&& pCurClient->GetKadState() == KS_NONE
				&& !pCurClient->GetDetailDialogInterface()->IsDialogOpen()	// NEO: MLD - [ModelesDialogs] <-- Xanatos --
				&& pCurClient->socket == NULL
				&& pCurClient->GetRequestFile() == NULL // NEO: FIXl <-- Xanatos --
#ifdef WEBCACHE // NEO: WC - [WebCache] -- Xanatos -->
				&& !pCurClient->IsProxy() //JP don't delete SINGLEProxyClient
#endif // NEO: WC END <-- Xanatos --
				)
			{
				cDeleted++;
				delete pCurClient;
			}
		}
		DEBUG_ONLY(AddDebugLogLine(false,_T("Cleaned ClientList, removed %i not used known clients"), cDeleted));
	}
}


CDeletedClient::CDeletedClient(const CUpDownClient* pClient)
{
	m_cBadRequest = 0;
	m_dwInserted = ::GetTickCount();
	//PORTANDHASH porthash = { pClient->GetUserPort(), pClient->Credits()};
	// NEO: XCTA - [XmanExtenedCreditTableArragement] -- Xanatos -->
	//make the Tracked-client-list independent 
	//track new port & hash
	PORTANDHASH porthash;
	porthash.nPort= pClient->GetUserPort();
	md4cpy(porthash.pHash,pClient->GetUserHash());
	// NEO: XCTA END <-- Xanatos --
	m_ItemsList.Add(porthash);
}

// ZZ:DownloadManager -->
void CClientList::ProcessA4AFClients() const {
    //if(thePrefs.GetLogA4AF()) AddDebugLogLine(false, _T(">>> Starting A4AF check"));
	for (POSITION pos = list.GetHeadPosition(); pos;){ // NEO: MOD - [LoopImprovement] <-- Xanatos --
		CUpDownClient* cur_client =	list.GetNext(pos); // NEO: MOD - [LoopImprovement] <-- Xanatos --
        if(cur_client->GetDownloadState() != DS_DOWNLOADING &&
           cur_client->GetDownloadState() != DS_CONNECTED &&
           (!cur_client->m_OtherRequests_list.IsEmpty() || !cur_client->m_OtherNoNeeded_list.IsEmpty())) {
            //AddDebugLogLine(false, _T("+++ ZZ:DownloadManager: Trying for better file for source: %s '%s'"), cur_client->GetUserName(), cur_client->reqfile->GetFileName());
               cur_client->SwapToAnotherFile(_T("Periodic A4AF check CClientList::ProcessA4AFClients()"), false, false, false, NULL, true, false);
        }
	}
    //if(thePrefs.GetLogA4AF()) AddDebugLogLine(false, _T(">>> Done with A4AF check"));
}
// <-- ZZ:DownloadManager

void CClientList::AddKadFirewallRequest(uint32 dwIP){
	IPANDTICS add = {dwIP, ::GetTickCount()};
	listFirewallCheckRequests.AddHead(add);
	while (!listFirewallCheckRequests.IsEmpty()){
		if (::GetTickCount() - listFirewallCheckRequests.GetTail().dwInserted > SEC2MS(180))
			listFirewallCheckRequests.RemoveTail();
		else
			break;
	}
}

bool CClientList::IsKadFirewallCheckIP(uint32 dwIP) const{
	for (POSITION pos = listFirewallCheckRequests.GetHeadPosition(); pos != NULL; listFirewallCheckRequests.GetNext(pos)){
		if (listFirewallCheckRequests.GetAt(pos).dwIP == dwIP && ::GetTickCount() - listFirewallCheckRequests.GetAt(pos).dwInserted < SEC2MS(180))
			return true;
	}
	return false;
}

// NEO: MIDS - [ModIDStat] -- Xanatos -->
void CClientList::GetModStatistics(CRBMap<uint32, CRBMap<CString, uint32>* > *clientMods){
	if (!clientMods)
		return;
	clientMods->RemoveAll();

	for (POSITION pos = list.GetHeadPosition(); pos;){ // NEO: MOD - [LoopImprovement]
		CUpDownClient* cur_client =	list.GetNext(pos); // NEO: MOD - [LoopImprovement]

		switch (cur_client->GetClientSoft()) {
		case SO_EMULE   :
		case SO_OLDEMULE:
			break;
		default:
			continue;
		}

		CRBMap<CString, uint32> *versionMods;

		if (!clientMods->Lookup(cur_client->GetVersion(), versionMods)){
			versionMods = new CRBMap<CString, uint32>;
			versionMods->RemoveAll();
			clientMods->SetAt(cur_client->GetVersion(), versionMods);
		}

		uint32 count;

		if (!versionMods->Lookup(cur_client->DbgGetFullClientSoftVer(), count))
			count = 1;
		else
			count++;

		versionMods->SetAt(cur_client->DbgGetFullClientSoftVer(), count);
	}
}

void CClientList::ReleaseModStatistics(CRBMap<uint32, CRBMap<CString, uint32>* > *clientMods){
	if (!clientMods)
		return;
	POSITION pos = clientMods->GetHeadPosition();
	while(pos != NULL)
	{
		uint32 version;
		CRBMap<CString, uint32> *versionMods;
		clientMods->GetNextAssoc(pos, version, versionMods);
		delete versionMods;
	}
	clientMods->RemoveAll();
}
// NEO: MIDS END <-- Xanatos --

// NEO: RIC - [ReaskOnIDChange] -- Xanatos -->
void CClientList::InformSourcesAboutIDChage()
{
	bool bNotDone = false;
	for(POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client =	list.GetNext(pos);
		if(thePrefs.IsReAskOnIPChange() && cur_client->GetRequestFile() != NULL) // The source will be informed on reask, no need to create extra overhead
			continue;

		if(cur_client->NotifyIdChage())
			bNotDone = true;
	}

	if(bNotDone)
		m_bNotifyIdChage = true;
}

void CClientList::ReAskAllClientsForDownload()
{
	for(POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client =	list.GetNext(pos);
		if(cur_client->GetRequestFile() == NULL)
			continue;

		// NEO: NSK - [NeoSourceKeeper] // NEO: NSS - [NeoSourceStorage] // NEO: XSC - [ExtremeSourceCache]
		if (cur_client->IsSurceSuspended())
			continue;
		// NEO: NSK END // NEO: NSS END // NEO: XSC END

		if (!cur_client->SetSafeReAskTime()) // Compute the next time that the file might be saftly reasked (=> no Ban)
			cur_client->SetDownloadState(DS_CONNECTINGWAITING);
	}	
}
// NEO: RIC END <-- Xanatos --

#ifdef IP2COUNTRY // NEO: IP2C - [IPtoCountry] -- Xanatos -->
void CClientList::ResetIP2Country(){

	CUpDownClient *cur_client;

	for(POSITION pos = list.GetHeadPosition(); pos != NULL; list.GetNext(pos)) { 
		cur_client = theApp.clientlist->list.GetAt(pos); 
		cur_client->ResetIP2Country();
	}

}
#endif // IP2COUNTRY // NEO: IP2C END <-- Xanatos --

#ifdef WEBCACHE // NEO: WC - [WebCache] -- Xanatos -->
// yonatan - not 2 be confused with the one in CUploadQueue!
CUpDownClient*	CClientList::FindClientByWebCacheUploadId(const uint32 id)
{
	for (POSITION pos = list.GetHeadPosition(); pos != NULL;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if ( cur_client->m_uWebCacheUploadId == id )
			return cur_client;
	}
	return 0;
}
// MORPH END - Added by Commander, WebCache 1.2e

// Superlexx - OHCB manager
// returns a list of multi-OHCB-supporting clients ( = v1.9a or newer ) that should receive this OHCB immediately
CUpDownClientPtrList* CClientList::XpressOHCBRecipients(uint32 maxNrOfClients, const Requested_Block_Struct* block)
{
	UINT part = (UINT)(block->StartOffset / PARTSIZE);
 	CUpDownClientPtrList* newClients = new CUpDownClientPtrList;	// supporting multi-OHCB

	for (POSITION pos = list.GetHeadPosition(); pos;)
	{
		CUpDownClient* cur_client = list.GetNext(pos);
		if ( !(cur_client->HasLowID() || (cur_client->socket && cur_client->socket->IsConnected()))
			&& cur_client->SupportsMultiOHCBs()
			&& !cur_client->IsProxy()	// client isn't a proxy
			&& cur_client->m_bIsAcceptingOurOhcbs
			//&& theApp.sharedfiles->GetFileByID(cur_client->GetUploadFileID())	// client has requested a file - obsolete with MFR
			//&& !md4cmp(cur_client->GetUploadFileID(), block->FileID ) // file hashes do match - obsolete with MFR
			&& !cur_client->IsPartAvailable(part, block->FileID) // the MFR version
			&& cur_client->IsBehindOurWebCache())	// inefficient
			if (cur_client->socket && cur_client->socket->IsConnected())
				newClients->AddHead(cur_client); // add connected clients to head
			else
				newClients->AddTail(cur_client);
	}

	CUpDownClientPtrList* toReturn = new CUpDownClientPtrList;

	// TODO: optimize this, dependent on further protocol development
	POSITION pos1 = newClients->GetHeadPosition();
	while (pos1 != NULL
		&& (UINT)toReturn->GetCount() <= maxNrOfClients)
		toReturn->AddTail(newClients->GetNext(pos1));

	delete newClients;
	return toReturn;
}

UINT CClientList::GetNumberOfClientsBehindOurWebCacheHavingSameFileAndNeedingThisBlock(Pending_Block_Struct* pending, UINT maxNrOfClients) // Superlexx - COtN
{
	UINT toReturn = 0;
	UINT part = (UINT)(pending->block->StartOffset / PARTSIZE);

	for (POSITION pos = list.GetHeadPosition(); pos && toReturn <= maxNrOfClients; list.GetNext(pos))
	{
		CUpDownClient* cur_client = list.GetAt( pos );
		if( cur_client->m_bIsAcceptingOurOhcbs
			&& !cur_client->IsProxy()
//			&& cur_client != this // 'this' is the client we want to download data from - covered by IsPartAvaiable
			&& cur_client->IsBehindOurWebCache()
			&& !cur_client->IsPartAvailable(part, pending->block->FileID))
			toReturn++;
	}
	return toReturn;
}

void CClientList::SendOHCBs()
{
	const uint32 now = ::GetTickCount();
	if (now - m_dwLastSendOHCBs > WC_SENDOHCBS_INTERVAL)
	{
		m_dwLastSendOHCBs = now;
		CUpDownClient* cur_client = NULL;
		for (POSITION pos = list.GetHeadPosition(); pos;)
		{
			cur_client = list.GetNext(pos);
			if (cur_client->SupportsWebCache()
				&& cur_client->SupportsMultiOHCBs()
				&& now - cur_client->lastMultiOHCBPacketSent > WC_MULTI_OHCB_SEND_TIME
				&& cur_client->IsBehindOurWebCache())
				cur_client->SendOHCBsNow();
		}
	}
}
#endif // NEO: WC END <-- Xanatos --